<?php

/**
 * This is the model class for table "Firma".
 *
 * The followings are the available columns in table 'Firma':
 * @property double $id
 * @property double $nazwa
 * @property string $czasUtworzenia
 * @property double $idDziedziny
 * @property double $stanKonta
 * @property string $nip
 * @property string $regon
 * @property string $urlLogo
 * @property int    $idBanku
 * @property double $idKlasy
 * @property double $status
 *
 * @property Biuro[] $biura
 * @property int $biuraIlosc
 * @property DziedzinaFirmy $dziedzina
 * @property Dziedzina $dziedzinaGlowna
 * @property Faktury $fakturies
 * @property Oplaty[] $oplaties
 * @property Oplaty[] $oplaty
 * @property Pracownik[] $pracowniks
 * @property int $workersCount
 * @property SzkolenieFirmy[] $szkolenieFirmies
 * @property Uzytkownik[] $uzytkownicy
 * @property Uzytkownik[] $activeUsers
 * @property ZlecenieFirmy[] $zlecenieFirmies
 * @property Klasa $klasa
 * @property PunktyFirmy $punktyFirmy
 * @property punktyDziedzinyFirmy[] $punktyDziedzinyFirmy
 * @property Bank $bank
 * @property SzkolenieFirmy $certyfikaty
 * @property Faktury[] $fakturyVat
 * @property Faktury[] $fakturyKosztowe
 * @property ZlecenieFirmy[] $notyKsiegowe
 * @property PracownikFirmy[] $activeWorkers
 */
class Firma extends CActiveRecord
{
	const COMPANY_STATUS_INACTIVE = 0;
	const COMPANY_STATUS_ACTIVE = 1;
	const COMPANY_STATUS_DEBT = 2;

	const ERROR_BANKRUT = 1230001;

	/* fields used in grid view filtering */
	public $className;
	public $schoolName;

	//helper field
	public $schoolId;

	public $updated = false;

	/**
	 * Returns the static model of the specified AR class.
	 * @param string $className active record class name.
	 * @return Firma the static model class
	 */
	public static function model($className=__CLASS__)
	{
		return parent::model($className);
	}

	/**
	 * @return string the associated database table name
	 */
	public function tableName()
	{
		return 'Firma';
	}

	/**
	 * @return array validation rules for model attributes.
	 */
	public function rules()
	{
		return array(
			array('nazwa', 'required'),
			array('idDziedziny, stanKonta, idKlasy, status', 'numerical'),
			array('nip', 'length', 'max'=>20),
			array('regon', 'length', 'max'=>18),
			array('urlLogo', 'length', 'max'=>512),
			array('schoolId', 'safe'),

			array('nip', 'ext.Validators.NipValidator'),
			array('regon', 'ext.Validators.Regon9Validator'),

			array('id, nazwa, czasUtworzenia, idDziedziny, stanKonta, nip, regon, urlLogo, className, schoolName, idBanku, status', 'safe', 'on'=>'search'),
		);
	}

	/**
	 * @return array relational rules.
	 */
	public function relations()
	{
		return array(
			'biura'=>array(self::MANY_MANY, 'Biuro', 'BiuroFirmy(idFirmy,idBiura)'),
			'biuraIlosc'=>array(self::STAT, 'Biuro', 'BiuroFirmy(idFirmy,idBiura)'),
			'dziedzina'=>array(self::HAS_MANY, 'DziedzinaFirmy', 'idFirmy'),
			'dziedzinaGlowna'=>array(self::BELONGS_TO, 'Dziedzina', 'idDziedziny'),
			'fakturies'=>array(self::HAS_MANY, 'Faktury', 'idFirmy'),
			'oplaties'=>array(self::HAS_MANY, 'Oplaty', 'idFirmy'),
			'oplaty'=>array(self::HAS_MANY, 'Oplaty', 'idFirmy',
				'order' => '"oplaty"."czas" DESC',
				'condition' => '"oplaty"."czas" IS NOT NULL OR "oplaty"."wartosc" <= 0'),
			'pracowniks'=>array(self::HAS_MANY, 'Pracownik', 'idFirmy'),
			'activeWorkers'=>array(self::HAS_MANY, 'PracownikFirmy', 'idFirmy',
				'condition' => '"activeWorkers"."dataZwolnienia" IS NULL'),
			'workersCount'=>array(self::STAT, 'Pracownik', 'idFirmy'),
			'szkolenieFirmies'=>array(self::HAS_MANY, 'SzkolenieFirmy', 'idFirmy'),
			'uzytkownicy'=>array(self::HAS_MANY, 'Uzytkownik', 'idFirmy'),
			'activeUsers'=>array(self::HAS_MANY, 'Uzytkownik', 'idFirmy', 'condition' => '"activeUsers"."status" = 1'),
			'zlecenieFirmies'=>array(self::HAS_MANY, 'ZlecenieFirmy', 'idFirmy'),
			'klasa'=>array(self::BELONGS_TO, 'Klasa', 'idKlasy'),
			'punktyFirmy'=>array(self::HAS_ONE, 'PunktyFirmy', 'idFirmy'),
			'punktyDziedzinyFirmy'=>array(self::HAS_MANY, 'PunktyDziedzinyFirmy', 'idFirmy'),
			'bank'=>array(self::BELONGS_TO, 'Bank', 'idBanku'),
			'certyfikaty'=>array(self::HAS_MANY, 'SzkolenieFirmy', 'idFirmy',
				'condition' => '"certyfikaty"."czySukces" > 0 AND "certyfikaty"."dataZakonczenia" <= SYSDATE',
				'with' => array('idSzkolenia0.dziedzina'),
				'order' => '"certyfikaty"."dataZakonczenia" DESC',
			),
			'fakturyVat'=>array(self::HAS_MANY, 'Faktury', 'idFirmy',
				'condition' => '"fakturyVat"."czyKoszt" = 0',
				'order' => '"fakturyVat"."dataPlatnosci" DESC',
			),
			'fakturyKosztowe'=>array(self::HAS_MANY, 'Faktury', 'idFirmy',
				'condition' => '"fakturyKosztowe"."czyKoszt" > 0',
				'order' => '"fakturyKosztowe"."dataPlatnosci" DESC',
			),
			'notyKsiegowe'=>array(self::HAS_MANY, 'ZlecenieFirmy', 'idFirmy',
				'condition' => '"notyKsiegowe"."czasPodjecia" IS NOT NULL AND
					"notyKsiegowe"."czasZakonczenia" IS NOT NULL AND
					"notyKsiegowe"."idOdpowiedzi" IS NULL',
				'order' => '"notyKsiegowe"."czasPodjecia" DESC',
				'with' => array('kontrahent', 'idZlecenia0.dziedzina')
			)
		);
	}

	/**
	 * @return array customized attribute labels (name=>label)
	 */
	public function attributeLabels()
	{
		return array(
			'id'=>Yii::t('Firma', 'ID'),
			'nazwa'=>Yii::t('Firma', 'Nazwa'),
			'czasUtworzenia'=>Yii::t('Firma', 'Utworzona'),
			'idDziedziny'=>Yii::t('Firma', 'Dziedzina'),
			'stanKonta'=>Yii::t('Firma', 'Stan konta'),
			'nip'=>Yii::t('Firma', 'NIP'),
			'regon'=>Yii::t('Firma', 'REGON'),
			'urlLogo'=>Yii::t('Firma', 'Logo'),
			'idBanku'=>Yii::t('Firma', 'Bank'),
			'schoolId'=>Yii::t('Firma', 'Szkoła'),
			'idKlasy'=>Yii::t('Firma', 'Klasa'),
		);
	}

	/**
	 * Retrieves a list of models based on the current search/filter conditions.
	 * @return CActiveDataProvider the data provider that can return the models based on the search/filter conditions.
	 */
	public function search()
	{
		$db = Yii::app()->db;

		$criteria = new CDbCriteria;
		$criteria->with = array('klasa.szkola','uzytkownicy');

		$classes = Yii::app()->user->getState('classes');
		if(Yii::app()->user->role == 'schoolAdmin' && !empty($classes))
			$criteria->addInCondition(Yii::app()->db->quoteTableName("t").'.'.Yii::app()->db->quoteColumnName("idKlasy"), Yii::app()->user->getState('classes'));
		elseif(Yii::app()->user->role == 'schoolAdmin' && empty($classes))
			return new CActiveDataProvider($this, array('data'=>array()));

		$criteria->compare($db->quoteTableName('t').'.'.$db->quoteColumnName('id'), $this->id);
		$criteria->compare($db->quoteTableName('t').'.'.$db->quoteColumnName('nazwa'), $this->nazwa, true);
		$criteria->compare($db->quoteColumnName('czasUtworzenia'), $this->czasUtworzenia, true);
		$criteria->compare($db->quoteColumnName('idDziedziny'), $this->idDziedziny);
		$criteria->compare($db->quoteColumnName('stanKonta'), $this->stanKonta);
		$criteria->compare($db->quoteColumnName('nip'), $this->nip, true);
		$criteria->compare($db->quoteColumnName('regon'), $this->regon, true);
		$criteria->compare($db->quoteColumnName('urlLogo'), $this->urlLogo, true);
		$criteria->compare($db->quoteTableName('t').'.'.$db->quoteColumnName('idKlasy'), $this->idKlasy);

		$criteria->compare($db->quoteTableName('klasa').'.'.$db->quoteColumnName('nazwa'), $this->className, true);
		$criteria->compare($db->quoteTableName('szkola').'.'.$db->quoteColumnName('nazwa'), $this->schoolName, true);

		return new CActiveDataProvider($this, array(
			'criteria'=>$criteria,
		));
	}

	public function getFriendlyName()
	{
		$name = trim($this->nazwa);
		return empty($name) ? 'bez nazwy' : $name;
	}

	public function getStanKontaFormatted()
	{
		return number_format($this->stanKonta, 2, ',', ' ');
	}

	public function getForms($id)
	{
		$sql = 'SELECT t."id", t0."nazwa", t."status"
			FROM "FormularzUzytkownika" t
			LEFT JOIN "Formularz" t0 ON t0."id" = t."idFormularza"
			WHERE t."idUzytkownika" IN (	SELECT t2."id"
							FROM "Uzytkownik" t2
							WHERE t2."idFirmy" = '.Yii::app()->db->quoteValue($id).')';

		return new CArrayDataProvider(Yii::app()->db->createCommand($sql)->queryAll());
	}

	public function getInactiveCompanies()
	{
		$sql = 'SELECT t2."idKlasy", t."id", t2."email", t2."pesel", t2."imie", t2."nazwisko", t."id" as "idFirmy", t3."nazwa" as "nazwaKlasy"
			FROM "Firma" t
			RIGHT JOIN "Uzytkownik" t2 ON t2."idFirmy"=t."id"
			LEFT JOIN "Klasa" t3 ON t3."id" = t2."idKlasy"
			WHERE t."status"=0 AND t."idKlasy" IN (SELECT t3."idKlasy"
								FROM "AdminKlasy" t3
								WHERE t3."idAdmina" = '.Yii::app()->user->id.')
			ORDER BY "idKlasy"';

		$result = array();
		$rawResult = Yii::app()->db->createCommand($sql)->queryAll();

		foreach($rawResult as $row)
		{
			$result[$row['idKlasy']][$row['idFirmy']][] = $row;
		}

		return $result;
	}

	public function getAllInactiveCompanies()
	{
		$sql = 'SELECT t2."idKlasy", t."id", t2."email", t2."pesel", t2."imie", t2."nazwisko", t."id" as "idFirmy", t3."nazwa" as "nazwaKlasy"
			FROM "Firma" t
			RIGHT JOIN "Uzytkownik" t2 ON t2."idFirmy"=t."id"
			LEFT JOIN "Klasa" t3 ON t3."id" = t2."idKlasy"
			WHERE t."status"=0
			ORDER BY "idKlasy"';

		$result = array();
		$rawResult = Yii::app()->db->createCommand($sql)->queryAll();

		foreach($rawResult as $row)
		{
			$result[$row['idKlasy']][$row['idFirmy']][] = $row;
		}

		return $result;
	}

	public function getAllStatistics($classId)
	{
		$sql = 'SELECT t5."nazwa", "idFirmy" as "id", "punkty"
			FROM "PunktyFirmy" t
			LEFT JOIN "Firma" t5 ON t5."id" = t."idFirmy"
			WHERE t."idFirmy" IN (SELECT t3."id"
						FROM "Firma" t3
						WHERE t3."idKlasy" = '.Yii::app()->db->quoteValue($classId).')
			ORDER BY "punkty" DESC';

			return new CSqlDataProvider($sql,array(
				'pagination'=>false
			));
	}

	public function getAllStatisticsTotal()
	{
		$sql = 'SELECT "idFirmy" as "id", t5."nazwa", "punkty", k."nazwa" AS "klasa", s."nazwa" AS "szkola"
		FROM "PunktyFirmy" t
		LEFT JOIN "Firma" t5 ON t5."id" = t."idFirmy"
		LEFT JOIN "Klasa" k ON k."id" = t5."idKlasy"
		LEFT JOIN "Szkola" s ON s."id" = k."idSzkoly"
		WHERE t."idFirmy" IN (SELECT t3."id"
			FROM "Firma" t3
			)';
		$count = Yii::app()->db->createCommand('SELECT COUNT(*) FROM "PunktyFirmy"')->queryScalar();

		$companies = Yii::app()->user->getState('companies');
		if(Yii::app()->user->role == 'schoolAdmin' && !empty($companies))
		{
			$sql = 'SELECT "idFirmy" as "id", t5."nazwa", "punkty", k."nazwa" AS "klasa", s."nazwa" AS "szkola"
			FROM "PunktyFirmy" t
			LEFT JOIN "Firma" t5 ON t5."id" = t."idFirmy"
			LEFT JOIN "Klasa" k ON k."id" = t5."idKlasy"
			LEFT JOIN "Szkola" s ON s."id" = k."idSzkoly"
			WHERE t."idFirmy" IN ('.implode(',', Yii::app()->user->getState('companies')).')';
		$count = Yii::app()->db->createCommand('SELECT COUNT(*) FROM "PunktyFirmy" WHERE "idFirmy" IN ('.implode(',', Yii::app()->user->getState('companies')).')')->queryScalar();
		}


		// return new CSqlDataProvider($sql,array(
		// 	'pagination'=>false
		// ));
		return new CSqlDataProvider($sql, array(
			'totalItemCount'=>$count,
			'pagination'=>array(
				'pageSize'=>100,
				)
			));
}

	public function getAllStatisticsByTeacher($classId,$teacherId)
	{
		$sql = 'SELECT t5."nazwa", "idFirmy" as "id", "punkty"
			FROM "PunktyFirmy" t
			LEFT JOIN "Firma" t5 ON t5."id" = t."idFirmy"
			WHERE t."idFirmy" IN (SELECT t3."id"
						FROM "Firma" t3
						WHERE t3."idKlasy" = '.Yii::app()->db->quoteValue($classId).')
				AND
				t5."idKlasy" IN (SELECT "AdminKlasy"."idKlasy" FROM "AdminKlasy" WHERE "AdminKlasy"."idAdmina" = '.$teacherId.')
			ORDER BY "punkty" DESC';

			return new CSqlDataProvider($sql,array(
				'pagination'=>false
			));
	}

	public function getPoints($companyId)
	{
		$sql = 'SELECT "punkty" FROM "PunktyFirmy" WHERE "idFirmy" = '.Yii::app()->db->quoteValue($companyId);
		return Yii::app()->db->createCommand($sql)->queryScalar();
	}

	public static function getValueRecords($classId)
	{
		$sql = '
			SELECT "x"."id", "x"."nazwa", "x"."dt", "x"."status", SUM("x"."wartosc") "wartosc" FROM
			(
				SELECT "f"."id", "f"."nazwa", 0 as "wartosc", 0 "status",
					("f"."czasUtworzenia" - to_date(\'01-01-1970\',\'DD-MM-YYYY\')) * (86400) as "dt"
				FROM "Firma" "f"
				WHERE "f"."idKlasy" = :idKlasy

				UNION ALL

				SELECT "f"."id", "f"."nazwa", nvl("f"."stanKonta", 0) "wartosc", 2 "status",
					(sysdate - to_date(\'01-01-1970\',\'DD-MM-YYYY\')) * (86400) as "dt"
				FROM "Firma" "f"
				WHERE "f"."idKlasy" = :idKlasy

				UNION ALL

				SELECT "f"."id", "f"."nazwa", "o"."wartosc", 1 "status",
					("o"."czas" - to_date(\'01-01-1970\',\'DD-MM-YYYY\')) * (86400) as "dt"
				FROM "Firma" "f"
				LEFT JOIN "Oplaty" "o" ON "o"."idFirmy" = "f"."id" AND "o"."czas" IS NOT NULL
				WHERE "f"."idKlasy" = :idKlasy AND "o"."id" IS NOT NULL
			) "x"
			GROUP BY "x"."id", "x"."nazwa", "x"."status", "x"."dt"
			ORDER BY "x"."id", "x"."dt"
		';
		$records = Yii::app()->db->createCommand($sql)->queryAll(true, array(
			':idKlasy' => $classId
		));
		$result = array();
		foreach ($records as $record) {
			$result[$record['id']][] = array(
				'czas' => $record['dt'],
				'wartosc' => $record['wartosc'],
				'nazwa' => $record['nazwa']
			);
		}
		$result2 = array();
		foreach ($result as $idx => $firma) {
			$tmp = array();
			for ($i = count($firma)-2; $i >= 0; $i--)
				$firma[$i]['wartosc'] = (float)$firma[$i+1]['wartosc'] - (float)$firma[$i]['wartosc'];
			for ($i = 0; $i < count($firma); $i++)
				$tmp[] = array((int)$firma[$i]['czas'] * 1000,(float)$firma[$i]['wartosc']);
			$firmaNazwa = isset($firma[0]['nazwa']) ? trim($firma[0]['nazwa']) : 'bez nazwy';
			$result2[] = array(
				'data' => $tmp,
				'label' => empty($firmaNazwa) ? 'bez nazwy' : $firmaNazwa,
				'color' => $idx,
			);
		}
		return $result2;
	}

	public static function getBusinessPointsInClassArray($classId)
	{
		$sql = '
			SELECT "f"."id", "f"."nazwa", "pf"."punkty" FROM "PunktyFirmy" "pf"
			LEFT JOIN "Firma" "f" ON "pf"."idFirmy" = "f"."id"
			WHERE "f"."idKlasy" = :idKlasy
			ORDER BY "pf"."punkty" DESC
			';
		return Yii::app()->db->createCommand($sql)->queryAll(true, array(
			':idKlasy' => $classId
		));
	}

	public static function getCompanyCertificatesCount($classId)
	{
		$sql = 'SELECT "f"."id", "f"."nazwa", "d"."nazwa" "dziedzina", COUNT("s"."id") "ilosc" FROM "Firma" "f"
			LEFT JOIN "Dziedzina" "d" ON "d"."id" IN (1, 2, 3, 4)
			LEFT JOIN "SzkolenieFirmy" "sf" ON "sf"."idFirmy" = "f"."id" AND "sf"."czySukces" > 0 AND "sf"."dataZakonczenia" <= SYSDATE
			LEFT JOIN "Szkolenie" "s" ON "sf"."idSzkolenia" = "s"."id" AND "s"."idDziedzina" = "d"."id"
			WHERE "f"."idKlasy" = :classId
			GROUP BY "f"."id", "f"."nazwa", "d"."nazwa"
			ORDER BY "f"."nazwa", "f"."id", "d"."nazwa"';
		$records = Yii::app()->db->createCommand($sql)->queryAll(true, array(
			':classId' => $classId
		));
		$result = array();
		foreach ($records as $record) {
			$result[$record['dziedzina']][$record['id']] = array(
				'nazwa' => $record['nazwa'],
				'ilosc' => $record['ilosc']
			);
		}
		return $result;
	}

	public function getAllStatisticsByClass($classId)
	{
		$sql = 'SELECT "punkty", "nazwa" ,t4."imie", t4."nazwisko", t."idFirmy" as "id"
			FROM "PunktyFirmy" t
			LEFT JOIN "Firma" t2 ON t2."id" = t."idFirmy"
			LEFT JOIN "AdminKlasy" t3 ON t2."idKlasy" = t3."idKlasy"
			LEFT JOIN "Admin" t4 ON t3."idAdmina" = t4."id"
			WHERE t2."idKlasy"='.Yii::app()->db->quoteValue($classId).'
			ORDER BY "punkty" DESC';

		return new CSqlDataProvider($sql,array(
			'pagination'=>false
		));
	}

	public function getMaximumStanKonta()
	{
		$command = Yii::app()->db->createCommand('
			select (
				select
				 max(nvl(f."stanKonta",0))
				from "Firma" f
				where f."idKlasy" = :idKlasy
			)
			* (
				SELECT pg."wartosc"
				FROM "ParametryGry" pg
				WHERE pg."nazwa" = \'DZIEDZINA_WAGA_STAN_KONTA\'
			)
			as maxi from dual');
		return $command->queryScalar(array(
			':idKlasy' => $this->idKlasy
		));
	}

	public function getStatistics()
	{

		$sql = '	SELECT t."nazwa", NVL(t3."punkty", 0) as "punkty"
				FROM "Dziedzina" t
				LEFT JOIN (SELECT t2."nazwa", t2."idFirmy", t2."idDziedziny", t2."punkty"
						FROM "PunktyDziedzinyFirmy" t2
						WHERE t2."idFirmy" = '.$this->id.') t3 ON t3."idDziedziny" = t."id"
			UNION ALL
				SELECT t4."nazwa", NVL(t4."punkty", 0) as "punkty"
				FROM "PunktyDziedzinyFirmy" t4
				WHERE t4."idFirmy" = '.$this->id.' AND t4."idDziedziny" IS NULL';

			return new CSqlDataProvider($sql,array(
				'keyField'=>'nazwa',
				'pagination'=>false
			));
	}

	public function generateNip()
	{
		$nip = 'PL285'.rand(0,9).rand(0,9).rand(0,9).rand(0,9).rand(0,9).rand(0,9).rand(0,9);

		return $nip;
	}

	public static function getTeachersBusiness($companyId)
	{
		$with = func_get_args(); unset($with[0]);
		return Firma::model()->with($with)->find(array(
			'join' => 'LEFT JOIN "AdminKlasy" "ak" ON "ak"."idKlasy" = "t"."idKlasy"',
			'condition' => '"t"."id" = :companyId AND "ak"."idAdmina" = :teacherId',
			'params' => array(
				':companyId' => $companyId,
				':teacherId' => Yii::app()->user->id
			)
		));
	}

	public static function getTeachersBusinessIds($companyIds)
	{
		$c = new CDbCriteria;
		$c->select = '"t"."id"';
		$c->join = 'LEFT JOIN "AdminKlasy" "ak" ON "ak"."idKlasy" = "t"."idKlasy"';
		$c->compare('"ak"."idAdmina"', Yii::app()->user->id);
		$c->addInCondition('"t"."id"', $companyIds);
		$ids = Firma::model()->findAll($c);
		$result = array();
		foreach ($ids as $rec)
			$result[] = $rec->id;
		return $result;
	}

	public static function getFreeSpace($companyId)
	{
		static $freeSpace = null;
		if (is_null($freeSpace))
			$freeSpace = (int)Yii::app()->db->createCommand('
				select nvl((
					select SUM("b"."maxPracownikow") from "BiuroFirmy" "bf"
					left join "Biuro" "b" on "b"."id" = "bf"."idBiura"
					where "bf"."idFirmy" = :id_firmy and "bf"."czasRezygnacji" is null
				),0) - (
					select count(*) from "PracownikFirmy" "pf"
					where "pf"."idFirmy" = :id_firmy and "pf"."dataZwolnienia" is null
				) as "freeSpace"
				from dual
			')->queryScalar(array(':id_firmy' => $companyId));
		return $freeSpace;
	}

	public static function getActiveTasksCount($companyId, $classId)
	{
		static $activeTasks = null;
		if (is_null($activeTasks))
			$activeTasks = (int)Yii::app()->db->createCommand('
				select count(*) from "ZlecenieFirmy"
				where "idFirmy" = :idFirmy AND "czasPodjecia" IS NOT NULL AND
				"idGrupyPrzetargowej" is NULL AND "czasPodjecia" >= (select "czasRozpoczecia" from (
					select * from "Tura" "t"
					where "t"."idKlasy" = :idKlasy
					order by "t"."czasRozpoczecia" DESC
				)
				where rownum <= 1)
			')->queryScalar(array(
				':idKlasy' => $classId,
				':idFirmy' => $companyId,
			));
		return $activeTasks;
	}

	public static function getTasksToGoCount($companyId)
	{
		static $tasksToGg = null;
		if (is_null($tasksToGg))
			$tasksToGg = (int)Yii::app()->db->createCommand('
				select count(*) from "ZlecenieFirmy" zf
        left join "PrzetargFirma" pf on zf."idGrupyPrzetargowej" = pf."idGrupy"
        left join "PrzetargGrupa" pg on zf."idGrupyPrzetargowej" = pg."id"
				where (zf."idUcznia" = :idUzytkownika or (pf."idFirmy" = :idFirmy and pf."status" > 0 and
					pg."status" > 1)) and zf."czasZakonczenia" is null
			')->queryScalar(array(':idFirmy' => $companyId, ':idUzytkownika' => Yii::app()->user->id));
		return $tasksToGg;
	}

	public static function getBonusTasksCount($companyId)
	{
		static $bonusTaskCount = null;
		if (is_null($bonusTaskCount))
			$bonusTaskCount = (int)Yii::app()->db->createCommand('
				select sum(k."bonus") from "KampaniaFirmy" kf
				left join "Kampanie" k on k."id" = kf."idKampanii"
				where kf."idFirmy" = :companyId and kf."czasZakonczenia" > sysdate
			')->queryScalar(array(':companyId' => $companyId));
		return $bonusTaskCount;
	}

	public static function getWorkForceCount($companyId)
	{
		static $baseTaskCount = null;
		if (is_null($baseTaskCount))
			$baseTaskCount = (int)Yii::app()->db->createCommand('select (
				select count(*) from "Uzytkownik" u
				left join "UzytkownikObecnosc" uo on uo."idUzytkownika" = u."id" and uo."idTury" = :lessonId
				where u."idFirmy" = :companyId and u."status" > 0 and (uo."id" is null or uo."obecny" > 0)
			) + (
				select count(*) from "PracownikFirmy" pf
				where pf."idFirmy" = :companyId and pf."dataZwolnienia" is null
			) "count" from dual')->queryScalar(array(
				':companyId' => $companyId,
				':lessonId' => Yii::app()->user->tura
			));
		return $baseTaskCount;
	}

	public static function getMaximumTasksCount($companyId)
	{
		return self::getWorkForceCount($companyId) + self::getBonusTasksCount($companyId);
	}

	public static function getLeftTasksCount($companyId, $classId)
	{
		return self::getMaximumTasksCount($companyId) - self::getActiveTasksCount($companyId, $classId);
	}

	public static function getTasksPerWorker($companyId)
	{
		$tasksCount = Firma::getMaximumTasksCount($companyId);
		$workersCount = Firma::getWorkForceCount($companyId);
		$result = ceil($tasksCount / $workersCount);
		if ($result < 1) $result = 1;
		return $result;
	}

	public static function getBonusTasksListCount($companyId)
	{
		static $baseTaskCount = null;
		if (is_null($baseTaskCount))
			$baseTaskCount = (int)Yii::app()->db->createCommand('
				select sum(k."bonus" + round((sysdate - kf."czasRozpoczecia") /
					(kf."czasZakonczenia" - kf."czasRozpoczecia") * k."przyrost")) "count"
				from "KampaniaFirmy" kf
				left join "Kampanie" k on k."id" = kf."idKampanii"
				where kf."idFirmy" = :companyId and kf."czasZakonczenia" > sysdate and k."id" > 0
			')->queryScalar(array(':companyId' => $companyId));
		return $baseTaskCount;
	}

	public static function getTasksListCount($companyId)
	{
		return self::getWorkForceCount($companyId) + self::getBonusTasksListCount($companyId);
	}

	public static function getDziedzinaReputation($companyId)
	{
		$results = Yii::app()->db->createCommand('
			select
				"z"."idDziedziny",
				sum(nvl2("t"."idOdpowiedzi", 1, 0)) / count(*) "ratio"
			from "ZlecenieFirmy" "t"
			left join "Zlecenie" "z" on "t"."idZlecenia" = "z"."id"
			where "t"."idFirmy" = :idFirmyZZ and "t"."czasZakonczenia" is not null and ("z"."wylaczone" is null or "z"."wylaczone" != 1)
			group by "z"."idDziedziny"
		')->queryAll(true, array(':idFirmyZZ' => $companyId));
		$return = array(1 => 1, 2 => 1, 3 => 1, 4 => 1);
		foreach ($results as $result)
			$return[(int)$result['idDziedziny']] = (float)$result['ratio'];
		return $return;
	}

	/**
	 * Pobiera maksymalny poziom dla każdej z dziedziny wynikających z certyfikató firmy lub pracownika
	 * @param int $companyId
	 * @param int $base domyslny poziom
	 * @return array
	 */
	public static function getMaxLevels($companyId, $base = 1)
	{
		$results = Yii::app()->db->createCommand('
			select "idDziedzina", max("poziom") "maxLvl" from (
			(
				select "s"."idDziedzina", "s"."poziom" from "SzkolenieFirmy" "sf"
				left join "Szkolenie" "s" on "sf"."idSzkolenia" = "s"."id"
				where "sf"."czySukces" > 0 and "sf"."idFirmy" = :idFirmy
			)
			union all
			(
				select "s"."idDziedzina", "s"."poziom" from "PracownikFirmy" "pf"
				left join "SzkoleniePracownika" "sp" on "pf"."idPracownika" = "sp"."idPracownika"
				left join "Szkolenie" "s" on  "s"."id" = "sp"."idSzkolenia"
				where "pf"."idFirmy" = :idFirmy and "sp"."id" is not null
			))
			group by "idDziedzina"
		')->queryAll(true, array(':idFirmy' => $companyId));
		$return = array(1 => $base, 2 => $base, 3 => $base, 4 => $base);
		foreach ($results as $result)
			$return[(int)$result['idDziedzina']] = (int)$result['maxLvl'];
		return $return;
	}

	/**
	 * aka getMaxLevels tylko ze certyfikaty firmy
	 * @param type $companyId
	 * @param type $base
	 * @return type
	 */
	public static function getCompanyMaxLevels($companyId, $base = 0)
	{
		$results = Yii::app()->db->createCommand('
			select "s"."idDziedzina", max("s"."poziom") "maxLvl" from "SzkolenieFirmy" "sf"
			left join "Szkolenie" "s" on "sf"."idSzkolenia" = "s"."id"
			where "sf"."czySukces" > 0 and "sf"."idFirmy" = :idFirmy
			group by "idDziedzina"
		')->queryAll(true, array(':idFirmy' => $companyId));
		$return = array(1 => $base, 2 => $base, 3 => $base, 4 => $base);
		foreach ($results as $result)
			$return[(int)$result['idDziedzina']] = (int)$result['maxLvl'];
		return $return;
	}

	public static function getReputations($companyId)
	{
		$maxLevels = self::getMaxLevels($companyId);
		$ratios = self::getDziedzinaReputation($companyId);
		return self::getReputationsArr($maxLevels, $ratios);
	}

	public static function getReputationsArr(array &$maxLevels, array &$ratios)
	{
		foreach ($maxLevels as $dziedzina => $level)
		{
			for ($l = 3; $l > 1; $l--)
				if ($level == $l && $ratios[$dziedzina] < ParametryGry::getValue('ZLECENIE_PROG_P'.$l))
					$level = $l - 1;
			$maxLevels[$dziedzina] = $level;
		}
		return $maxLevels;
	}

	/**
	 * wysyla e-mail z dodatkowymi zleceniami do firmy
	 * @param type $companyId
	 */
	public function sendBonusTasks()
	{
		//czy można pobierac dodatkowe zlecenia (kampania)
		$bonusTasks = self::getBonusTasksListCount($this->id);
		if ($bonusTasks <= 0)
			return false;

		$actions = Akcja::model()->findAll(array(
			'join' => 'left join "AkcjaUzytkownika" au on au."idAkcji" = "t"."id"',
			'condition' => 'au."id" is null and "t"."idLekcji" = :lessonId and "t"."nazwa" = :tag',
			'params' => array(
				':lessonId' => Yii::app()->user->tura,
				':tag' => Akcja::ACTION_MAIL,
			)
		));
		if (empty($actions))
			return false;

		$patterns = array('/\[foreignLink\]/i', '/\[bonusoweZlecenia\]/i');
		$replacements = array(
			CHtml::link('here', '#', array('class'=>'bonusTasksLink','data-lang'=>'en')),
			CHtml::link('tutaj', '#', array('class'=>'bonusTasksLink','data-lang'=>'')),
		);

		foreach ($actions as $action)
		{
			/* @var Akcja $action */

			$title = preg_replace($patterns, $replacements, Akcja::parseText($action->naglowek));
			$content = preg_replace($patterns, $replacements, Akcja::parseText($action->tresc));

			foreach ($this->activeUsers as $user)
			{
				$message = new AkcjaUzytkownika;
				$message->idAkcji = $action->id;
				$message->idUzytkownika = $user->id;
				$message->dataWyslania = new CDbExpression('SYSDATE');
				$message->przeczytana = 0;
				$message->naglowek = $title;
				$message->tresc = $content;
				$message->save(false);
			}
		}

		return true;
	}

	public static function getWorkersList($companyId)
	{
		$tura = Tura::getTuraByCompany($companyId);
		if (empty($tura))
			throw new Exception('Żadna lekcja nie została jeszcze rozpoczęta.');
		$tasksPerWorker = Firma::getTasksPerWorker($companyId);

		$workers = Yii::app()->db->createCommand('(
			select \'w\'||p."id" "id", p."imie"||\' \'||p."nazwisko"||\' (\'||count(zf."id")||\'/\'||:tasksPerWorker||\')\' "name", \'Pracownicy\' "type" from "PracownikFirmy" pf
			left join "Pracownik" p on p."id" = pf."idPracownika"
      left join "ZlecenieFirmy" zf on zf."idPracownika" = p."id" and zf."idFirmy" = :companyId and zf."czasPodjecia" >= :periodStart and zf."czasZakonczenia" is null
			where pf."idFirmy" = :companyId and pf."dataZwolnienia" is null
      group by p."id", p."imie", p."nazwisko"
		) union all (
			select \'s\'||u."id" "id", u."imie"||\' \'||u."nazwisko"||\' (\'||count(zf."id")||\'/\'||:tasksPerWorker||\')\' "name", \'Uczniowie\' "type" from "Uzytkownik" u
			left join "UzytkownikObecnosc" uo on uo."idUzytkownika" = u."id" and uo."idTury" = :lessonId
      left join "ZlecenieFirmy" zf on zf."idUcznia" = u."id" and zf."idFirmy" = :companyId and zf."czasPodjecia" >= :periodStart and zf."czasZakonczenia" is null
			where u."status" > 0 and u."idFirmy" = :companyId and nvl(uo."obecny", 1) > 0
      group by u."id", u."imie", u."nazwisko"
		) order by "type" desc, "name" asc
		')->queryAll(true, array(
			':companyId' => $companyId,
			':lessonId' => $tura->id,
			':periodStart' => $tura->czasRozpoczecia,
			':tasksPerWorker' => $tasksPerWorker
		));
		$workersList = array('' => '') + CHtml::listData($workers, 'id', 'name', 'type');
		return $workersList;
	}

	public function alayzePayments($lessonNumber)
	{
/*		$akcja = Akcja::model()->with('userAction')->find(array(
			'condition' => '"t"."idLekcji" = :turaId and "t"."nazwa" = :tag',
			'params' => array(
				':userId' => Yii::app()->user->id,
				':tag' => Akcja::ACTION_PAYMENTS,
				':turaId' => Yii::app()->user->tura
			)
		));
		if (empty($akcja))
			return false;
*/
		$status = self::COMPANY_STATUS_ACTIVE;

		//oplata za biuro
		$offises = Oplaty::analyzeBiuroPlayments($this);
		if (!empty($offises))
			$status = self::COMPANY_STATUS_DEBT;

		//oplata za banbk
		$bank = Oplaty::analyzeBankPlayments($this, $lessonNumber);

		//oplata za noty ksiegowe
		//oplata za faktury kosztowe
/*		$kontrahenci = Oplaty::analyzeKontrahentPayments($this);
		if (!empty($kontrahenci))
			$status = self::COMPANY_STATUS_DEBT;
*/
		//wyplaty pracownikow (w tym zus)
		$payemts = Oplaty::analyzeSalaryPayments($this);
		if (!empty($payemts))
			$status = self::COMPANY_STATUS_DEBT;

		//podatek dochodowy
		$tax = Oplaty::analyzeTaxPayments($this, $lessonNumber);
		if (!empty($tax))
			$status = self::COMPANY_STATUS_DEBT;

		//vat
		$vat = Oplaty::analyzeVatPayments($this, $lessonNumber);
		if (!empty($vat))
			$status = self::COMPANY_STATUS_DEBT;

		if(empty($akcja->userAction) && $status == self::COMPANY_STATUS_DEBT)
		{
			$all = $vat + $tax + $payemts + $bank + $offises;

			$list = '<br /><ul style="margin-bottom: 0">';

			// kapitalizacja odsetek oprocentowanie i koszty bankowe

			foreach($bank as $record)
				$list .= '<li>'.$record['who'].' - '.$record['details'].' - '.$record['value'].' zł</li>';

			foreach($offises as $record)
			{
				//$list .= '<li>'.$record['who'].' - '.$record['details'].' - '.$record['value'].' zł</li>';

				$list .= '<li>'.$record['who'].'<br/> - <a href="#" class="paylink payOffice" data-kontrahentid="209" data-idfirmy="'.$this->id.
				'" data-rodzajoplaty="'.Oplaty::TYPE_OFFICE.'" data-title="'.$record['details'].'" data-wartosc="'.number_format($record['value'],2,".","").'">'.$record['details'].
				' - <span style="color: #FF0000;">'.$record['value'].'</span> zł [zapłać]</a></li>';
			}
/*
			foreach($kontrahenci as $record)
				$list .= '<li>'.$record['who'].' - '.$record['details'].' - '.$record['value'].' zł</li>';
*/

			// ZUS przedsiebiorcy
			$ZPparamUS = ParametryGry::getFloat('ZUS_US_PRZEDSIEBIORCY', 160.78);
                	$ZPparamUZ = ParametryGry::getFloat('ZUS_UZ_PRZEDSIEBIORCY', 270.40);

			$list .= '<li>
			ZUS przedsiebiorcy: <br/>
			- <a href="#" class="paylink payTax" data-kontrahentid="148" data-idfirmy="'.$this->id.
			'" data-rodzajoplaty="'.Oplaty::TYPE_ZUS_US.'" data-title="Ubezpieczenie Społeczne przedsiębiorcy NIP: '.$this->nip.'" data-wartosc="'.number_format($ZPparamUS,2,".","").'">
			Ubezpieczenie Społeczne - <span style="color: #FF0000;">'.$ZPparamUS.'</span> zł [zapłać]</a><br/>
			- <a href="#" class="paylink payTax" data-kontrahentid="148" data-idfirmy="'.$this->id.
			'" data-rodzajoplaty="'.Oplaty::TYPE_ZUS_UZ.'" data-title="Ubezpieczenie Zdrowotne przedsiębiorcy NIP: '.$this->nip.'" data-wartosc="'.number_format($ZPparamUZ,2,".","").'">
			Ubezpieczenie Zdrowotne - <span style="color: #FF0000;">'.$ZPparamUZ.'</span> zł [zapłać]</a>
			</li>';

			$pensjaMinimalna     = ParametryGry::getFloat('PENSJA_MINIMALNA', 1680.00);
			$emeryturaPracodawca = ParametryGry::getFloat('ZUS_EMER_PRACODAWCA_PROC', 9.76);
			$emeryturaPracownik  = ParametryGry::getFloat('ZUS_EMER_PRACOWNIK_PROC', 9.76);
			$rentaPracodawca     = ParametryGry::getFloat('ZUS_RENT_PRACODAWCA_PROC', 6.50);
			$rentaPracownik      = ParametryGry::getFloat('ZUS_RENT_PRACOWNIK_PROC', 1.50);
			$wypadkowePracodawca = ParametryGry::getFloat('ZUS_WYP_PRACODAWCA_PROC', 1.93);
			$chorobowePracownik  = ParametryGry::getFloat('ZUS_CHOR_PRACOWNIK_PROC', 2.45);
			$zdrowotnePracownik  = ParametryGry::getFloat('ZUS_ZDROW_PRACOWNIK_PROC', 9.00);
			$funduszPracodawca   = ParametryGry::getFloat('ZUS_FP_PRACODAWCA_PROC', 2.45);
			$kosztUzyskPrzychodu = ParametryGry::getFloat('ZUS_KOSZT_UZYSKANIA_PRZYCHODU', 111.25);
			$podatekDochodowy    = ParametryGry::getFloat('PODATEK_DOCHODOWY', 0.18);

			foreach($payemts as $record)
			{
				$brutto = $record['value'];
				$ubezpSpoleczne = $brutto*($emeryturaPracownik/100.0) + $brutto*($rentaPracownik/100.0) + $brutto*($chorobowePracownik/100.0);
				$ubezpZdrowotne = ($brutto-$ubezpSpoleczne)*($zdrowotnePracownik/100.0);
				$podatekZaPracownika = round($brutto-$ubezpSpoleczne-$kosztUzyskPrzychodu)*$podatekDochodowy;
				$zaliczkaPodatku = ($podatekZaPracownika<=$ubezpZdrowotne)? 0 : ($podatekZaPracownika - ($brutto-$ubezpSpoleczne)*(7.75/100.0));
				$kosztPracownika = $ubezpSpoleczne+$ubezpZdrowotne+$zaliczkaPodatku;
				$netto = $brutto - $kosztPracownika;

				$spolecznePracodawca = $brutto*($emeryturaPracodawca/100.0) + $brutto*($rentaPracodawca/100.0) + $brutto*($wypadkowePracodawca/100.0) + $brutto*($funduszPracodawca/100.0);

				//$list .= '<li>'.$record['who'].' - '.$record['details'].' - '.$record['value'].' zł</li>';
				$list .= '<li>'.$record['who'].
				'<br/> - <a href="#" class="paylink payTax" data-kontrahentid="210" data-idfirmy="'.$this->id.
				'" data-rodzajoplaty="'.Oplaty::TYPE_WORKER.'" data-title="'.$record['details'].' '.$record['who'].'" data-wartosc="'.number_format($netto,2,".","").'">'.$record['details'].
				' - <span style="color: #FF0000;">'.number_format($netto,2,",","").'</span> zł netto [zapłać]</a></li>';

				// ZUS pracownikow
				$list .= '<li>
				ZUS pracownika: <br/>
				- <a href="#" class="paylink payTax" data-kontrahentid="148" data-idfirmy="'.$this->id.
				'" data-rodzajoplaty="'.Oplaty::TYPE_ZUS_US.'" data-title="Ubezpieczenie Społeczne: '.$this->nip.'" data-wartosc="'.number_format($ubezpSpoleczne+$spolecznePracodawca,2,".","").'">
				Ubezpieczenie Społeczne - <span style="color: #FF0000;">'.number_format($ubezpSpoleczne+$spolecznePracodawca,2,",","").'</span> zł [zapłać]</a><br/>
				- <a href="#" class="paylink payTax" data-kontrahentid="148" data-idfirmy="'.$this->id.
				'" data-rodzajoplaty="'.Oplaty::TYPE_ZUS_UZ.'" data-title="Ubezpieczenie Zdrowotne: '.$this->nip.'" data-wartosc="'.number_format($ubezpZdrowotne,2,".","").'">
				Ubezpieczenie Zdrowotne - <span style="color: #FF0000;">'.number_format($ubezpZdrowotne,2,",","").'</span> zł [zapłać]</a>
				</li>';

				// podatek za pracownika
				$list .= '<li>
				Zaliczka podatku za pracownika - <a href="#" class="paylink payTax" data-kontrahentid="189" data-idfirmy="'.$this->id.'" data-rodzajoplaty="'.Oplaty::TYPE_TAX.'" data-title="Zaliczka podatku za pracownika" data-wartosc="'.number_format($zaliczkaPodatku,2,".","").'"> <span style="color: #FF0000;">'.number_format($zaliczkaPodatku,2,",","").'</span> zł [zapłać]</a>
				</li>';
			}


			foreach($tax as $record)
			{
				if($record['value']>0)
					$list .= '<li>'.$record['who'].
					'<br/> - <a href="#" class="paylink payTax" data-kontrahentid="189" data-idfirmy="'.$this->id.
					'" data-rodzajoplaty="'.Oplaty::TYPE_TAX.'" data-title="'.$record['details'].'" data-wartosc="'.number_format($record['value'],2,".","").'">'.$record['details'].
					' - <span style="color: #FF0000;">'.number_format($record['value'],2,",","").'</span> zł [zapłać]</a></li>';
				else if($record['value']<0)
					$list .= '<li>'.$record['who'].' - '.$record['details'].' (zwrot) - <span style="color: #00FF00;">'.-number_format($record['value'],2,",","").'</span> zł</li>';
			}

			foreach($vat as $record)
			{
				if($record['value']>0)
					$list .= '<li>'.$record['who'].'<br/> - <a href="#" class="paylink payVat" data-kontrahentid="189" data-idfirmy="'.$this->id.
					'" data-rodzajoplaty="'.Oplaty::TYPE_VAT.'" data-title="'.$record['details'].'" data-wartosc="'.number_format($record['value'],2,".","").'">'.$record['details'].
					' - <span style="color: #FF0000;">'.number_format($record['value'],2,",","").'</span> zł [zapłać]</a></li>';
				else
					$list .= '<li>'.$record['who'].' - '.$record['details'].' - <span style="color: #00FF00;">'.-number_format($record['value'],2,",","").'</span> zł</li>';
			}

			$list .= '</ul>';

			return $list;
/*
			foreach ($this->activeUsers as $user)
			{
				$msg = $akcja->generateFromUserModel($user, false);
				$msg->tresc = str_replace('[lista]', $list, $msg->tresc);
				//$msg->save(false);
			}*/
		}
/*
		if ($status != $this->status)
		{
			$this->status = $status;
			$this->updated = true;
		}
*/
		//return $this->updated;
		return '';
	}

	public function getUserNames($withCompanyName = false)
	{
		$userNames = array();

		foreach($this->uzytkownicy as $uzytkownik)
			$userNames[] = $uzytkownik->nazwisko.' '.$uzytkownik->imie;

		return implode(', ', $userNames).($withCompanyName && !empty($this->nazwa) ? ' ('.$this->nazwa.')' : '');
	}

	public function makePayment($value, $kontrahentId, $title)
	{
		$oplata = new Oplaty;
		$oplata->rodzajOplaty = Oplaty::TYPE_TASK_PAYMENT;
		$oplata->idFirmy = $this->id;
		$oplata->idKontrahenta = $kontrahentId;
		$oplata->tytul = $title;
		$oplata->wartosc = -abs($value);
		$oplata->czas = new CDbExpression('SYSDATE');
		$oplata->save(false);
		$this->stanKonta += $oplata->wartosc;

		if ($this->bank->oplataPrzelew > 0)
		{
			$bank = new Oplaty;
			$bank->rodzajOplaty = Oplaty::TYPE_BANK_TRANSFER;
			$bank->idFirmy = $this->id;
			$bank->idBanku = $this->idBanku;
			$bank->tytul = 'Opłata za przelew.';
			$bank->wartosc = -$this->bank->oplataPrzelew;
			$bank->czas = new CDbExpression('SYSDATE');
			$bank->save(false);
			$this->stanKonta += $bank->wartosc;
		}

		Yii::app()->user->setState('bankrut', $this->stanKonta <= 0);

		return $oplata->id;
	}

	public function payOrder(Faktury &$faktura)
	{
		if ($this->idBanku <=0 || $this->idBanku >4)
			throw new Exception('Nie masz jeszcze konta w banku!');

		if (!$faktura->czyKoszt)
			throw new Exception('Faktura nie jest kosztowa.');

		Yii::app()->user->setState('bankrut', $this->stanKonta <= 0);

		if ($this->stanKonta < $faktura->cenaBrutto + $this->bank->oplataPrzelew)
			throw new Exception('Brak wystarczających środków na koncie.');

		$this->makePayment($faktura->cenaBrutto, $faktura->idKontrahenta,
			'Opłata za fakturę ' . $faktura->nrFaktury);
		$faktura->dataPlatnosci = date('d-m-Y H:i:s');
	}

	public function transferPoints($dziedzinaId, $points)
	{
		DziedzinaFirmy::add($this->id, $dziedzinaId, $points);
		foreach ($this->activeUsers as $user)
			DziedzinaUzytkownika::add($user->id, $dziedzinaId, $points);
	}

	public function transferPointsWithWorkers($dziedzinaId, $points)
	{
		$this->transferPoints($dziedzinaId, $points);
		foreach ($this->activeWorkers as $worker)
			DziedzinaPracownika::add($worker->idPracownika, $dziedzinaId, $points);
	}
}
